#!/usr/bin/env python3
# coding: utf-8
##############################################################################
#
# @brief Spec file that used to collect all data files for corresponding package
# All other configuration options will be taken from pyproject.toml
#
# @author dmitry.ryzhov@cloudbear.ru
#
# Copyright (c) 2025 CloudBEAR LLC - http://www.cloudbear.ru/
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#   Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in the
#   documentation and/or other materials provided with the
#   distribution.
#
#   Neither the name of CloudBEAR nor the names of
#   its contributors may be used to endorse or promote products derived
#   from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
##############################################################################

import collections
import glob
import os
import setuptools

try:
    import tomllib  # Python 3.11+
except ModuleNotFoundError:
    import tomli as tomllib  # backport for 3.9

with open('pyproject.toml', 'rb') as f:
    config = tomllib.load(f)
package_name = config['project']['name']
data_target_dir = config.get('tool', {}).get('bear-pipbuildkit', {}).get(
    'data-target-dir', package_name)
data_patterns = config.get('tool', {}).get('bear-pipbuildkit', {}).get('data-patterns', [])
scripts_patterns = config.get('tool', {}).get('bear-pipbuildkit', {}).get('scripts-patterns', [])


def get_data_files(target_dir, patterns):
    data_files_dict = collections.defaultdict(list)
    for pattern in patterns:
        matching_files = glob.glob(pattern, recursive=True)
        if not matching_files:
            print(f'Warning: No matches found for pattern "{pattern}"')
        for filepath in matching_files:
            if os.path.isfile(filepath):
                rel_path = os.path.relpath(filepath, '.')
                install_dir = os.path.dirname(rel_path)
                if target_dir:
                    install_dir = os.path.join(target_dir, install_dir)
                data_files_dict[install_dir].append(filepath)
            else:
                print(f'Warning: {filepath} is not a file and will be skipped.')

    data_files = list(data_files_dict.items())
    return data_files


def get_scripts(patterns):
    scripts = []
    for pattern in patterns:
        scripts.extend(glob.glob(pattern))

    return scripts


setuptools.setup(
    data_files=get_data_files(data_target_dir, data_patterns),
    scripts=get_scripts(scripts_patterns),
)
